# ======================================================================
#   convert_to_webp.py
#   （JPG / PNG / GIF / BMP / PDF を WebP に変換するツール）
#======================================================================
#   convert_to_webp.py
#   （指定フォルダ内の JPG / PNG / GIF / BMP / PDF を WebP化するツール）
#======================================================================

#【概要】
#指定したフォルダ内にある画像（JPG / JPEG / PNG / GIF / BMP）と PDF を
#自動的に WebP 形式へ変換し、別フォルダへ保存する Python ツールです。

#PDF は 1ページ目を自動レンダリングし、画像化してから WebP 化します。
#画像は Pillow、PDF は pypdfium2 を使用しています。

#出力ファイル名の形式：
#    <元ファイル名>_YYYYMMDD_連番.webp
#    2025/12/27 日付と連番はつけたくないため変更
#例：
#    sample_20250212_01.webp
#    <元ファイル名>.webp（※日付・連番なし）

#対象者：
#  ・Web用サムネイルを大量生成したい人
#  ・JPG/PNG/PDF を軽量な WebP に統一したい人
#  ・手動で Photoshop 等を開くのが面倒な人

#======================================================================

#【実行方法】
#PowerShell または CMD より次のいずれかで実行します。

#  python convert_to_webp.py
#  py convert_to_webp.py

#同じフォルダに移動してから実行してください。

#起動後はフォルダ選択ダイアログが2回表示されるので、
#「入力フォルダ」「出力フォルダ」を順に選んでください。

#======================================================================

#【必要なライブラリ（初回のみインストール）】
#PowerShell または CMD で次を実行：
#
#    pip install pillow pypdfium2
#
#※ Python が導入済みである必要があります。

#======================================================================

#【作成の経緯】
#・Webサイト（tentenkoubo.com）の更新に使用する画像を
# jpg / png / pdf から webp に自動変換する必要があったため。
#・PowerShell + cwebp バージョンは運用に問題なかったが、
# PDF にも対応した「完全自動処理版」が求められたため。
#・Python なら PDF → 画像 → WebP を一気通貫できるため、
# 本ツールを作成した。

#======================================================================
# 【フォルダ名に関する注意事項】
#
# Windows のパスでは が区切りとして使われますが、
# Python は文字列内の [\] を「エスケープシーケンス」と解釈します。
# 
# 例：
#   C:\Users[\]admin\Desktop\ショートカット\甜々工房\テンテン工房.com[\]tools
#
# 上記のように「\」の後に日本語が来る場合（例：\甜, \テ, \〇）、
# Python はこれを無効なエスケープとして扱い、
#
#   SyntaxWarning: invalid escape sequence
#
# の警告を表示します。
#
# ※ この警告はツールの動作には全く影響しません。
# ※ あくまで Python が「文字列の中にある \◯ を誤解した」だけです。
#
# ======================================================================
# 【本ツールでの対応】
#
# Python がエスケープを誤判定しないように、
# 以下のように rに変更しています。
#
# r""" 
#       （ドキュメントコメント）
#
# これにより、docstring 内の \ を Python が解釈せず、
# invalid escape sequence の警告が確実に抑制されます。
#
# ======================================================================
# 【推奨運用】
#
# ・Python ファイル名に日本語・記号（括弧など）・空白を使わない
# ・フォルダ名に「\ + 日本語」の組み合わせを避ける
# ・どうしても必要な場合は本ツールのように r を利用する
#
# ======================================================================

# 【追加仕様】
#   ・PDF は全ページを JPG に書き出し、その JPG を WebP 化します。
#   ・同名の WebP が存在する場合は自動スキップします。
#   ・最終的に「何ファイル作成したか」を表示します。
#
# 【注意（エスケープ警告について）】
#   Windows パスに含まれる「\ + 日本語」（例：\甜、\〇）が
#   Python により "Unicode エスケープ" と誤判定される場合があります。
#   このファイルではコメント内にパスを直接書かず、警告が出ないよう配慮しています。
#
# ======================================================================
# import os   #システム日付を取得しないため削除　2025/12/27
#import datetime
#↑ファイル名に日付をつけなくてよいため削除　　2025/12/27
from pathlib import Path
from tkinter import Tk, filedialog
from PIL import Image
import pypdfium2 as pdfium


# ----------------------------------------------------------------------
# PDF → 全ページ JPG に変換する関数
# ----------------------------------------------------------------------
def pdf_to_images(pdf_path, input_dir, base_stem):
    """
    PDF を全ページ JPG に変換し、
    PDF と同じフォルダ（input_dir）に保存して、
    JPG のパスを list で返す
    """
    pdf = pdfium.PdfDocument(pdf_path)
    total = len(pdf)
    jpg_paths = []

    for page_number in range(total):
        page = pdf.get_page(page_number)
        bitmap = page.render(scale=2)
        img = bitmap.to_pil()

        # JPG は PDF と同じフォルダに書き出す
        jpg_name = f"{base_stem}_p{page_number+1:02d}.jpg"
        jpg_path = input_dir / jpg_name
        img.save(jpg_path, "JPEG", quality=95)

        jpg_paths.append(jpg_path)

        page.close()

    pdf.close()
    return jpg_paths



# ----------------------------------------------------------------------
# WebP 保存用
# ----------------------------------------------------------------------
def save_webp(input_image, output_path):
    input_image.save(output_path, "webp", quality=90, method=6)


# ----------------------------------------------------------------------
# メイン処理
# ----------------------------------------------------------------------
def main():

    # GUI ダイアログ非表示設定
    Tk().withdraw()

    print("入力フォルダ（JPG/PNG/PDF など）を選んでください")
    input_dir = filedialog.askdirectory()
    if not input_dir:
        print("中断しました。")
        return

    print("出力フォルダ（WebP 保存先）を選んでください")
    output_dir = filedialog.askdirectory()
    if not output_dir:
        print("中断しました。")
        return

    input_dir = Path(input_dir)
    output_dir = Path(output_dir)

    valid_exts = [".jpg", ".jpeg", ".png", ".gif", ".bmp", ".pdf"]

    print("\n----- 変換開始 -----\n")

    created_count = 0  # 作成した WebP の数

    for file in input_dir.iterdir():
        if not file.is_file():
            continue

        ext = file.suffix.lower()
        if ext not in valid_exts:
            print(f"対象外：{file.name}")
            continue

        # ----------------------------------------
        # PDF の場合：全ページ JPG → WebP
        # ----------------------------------------
        if ext == ".pdf":
            print(f"[PDF] {file.name} → ページ展開中…")

            jpg_files = pdf_to_images(str(file), input_dir, file.stem)

            for jpg_path in jpg_files:
                webp_name = jpg_path.stem + ".webp"
                webp_path = output_dir / webp_name

                if webp_path.exists():
                    print(f"スキップ：{webp_name}（既に存在）")
                    continue

                img = Image.open(jpg_path)
                save_webp(img, webp_path)
                print(f"✔ 完了 → {webp_name}")

                created_count += 1

            continue

        # ----------------------------------------
        # 画像ファイルの場合：そのまま WebP 化
        # ----------------------------------------
        print(f"[画像] {file.name} → WebP 生成中…")
        webp_name = f"{file.stem}.webp"
        webp_path = output_dir / webp_name

        if webp_path.exists():
            print(f"スキップ：{webp_name}（既に存在）")
            continue

        image = Image.open(file)
        save_webp(image, webp_path)

        print(f"✔ 完了 → {webp_name}")
        created_count += 1

    # ------------------------------------------------------------------
    # 終了メッセージ
    # ------------------------------------------------------------------
    print("\n----- 変換終了 -----")
    print(f"{created_count} 件の WebP を新規作成しました。")
    input("Enterキーで閉じます…")


# ----------------------------------------------------------------------
# 実行
# ----------------------------------------------------------------------
if __name__ == "__main__":
    main()
