# -*- coding: utf-8 -*-
"""
======================================================================
   YouTube ダウンローダー（改良版 / MP4 音声互換性強化）
======================================================================

【概要】
YouTube の動画を MP4 / MP3 形式で保存するツールです。
音声形式が Opus の場合、Windows 標準アプリでは音声が出ないため、
互換性の高い AAC への変換を試みます。

======================================================================
【事前に必要なもの（重要！）】
このツールは以下の外部コマンドを使用します。

 1. yt-dlp  
    最新版ダウンロード：
        https://github.com/yt-dlp/yt-dlp/releases/latest/download/yt-dlp.exe
    → 任意フォルダに置き、PATH を通す、または同じフォルダに置く。

 2. ffmpeg  
    Windows版ダウンロード（ZIP）：
        https://www.gyan.dev/ffmpeg/builds/
    → 解凍後、ffmpeg.exe が存在するフォルダを PATH に追加する。

◆ PATH を通さない場合
yt-dlp.exe と ffmpeg.exe を、このツール（.py）のあるフォルダに置けば動きます。

◆ インストール確認方法（PowerShell）
    yt-dlp --version
    ffmpeg -version

どちらもバージョンが表示されれば OK です。

======================================================================
【音声が再生されない場合について】

YouTube の動画には音声が「Opus」形式のものが多くあります。
Windows 標準アプリ（映画＆テレビ、フォト）は Opus をサポートしていないため、
MP4 を再生しても「映像のみで音声が出ない」場合があります。

これはツールの不具合ではなく、
YouTube の配信仕様と Windows 標準プレーヤー側の制限によるものです。

■ 音声を確実に再生するには以下のプレーヤーを推奨します。
   - VLC media player
   - MPC-HC
   - mpv player

======================================================================
【保存先】
※任意の保存先で問題ありません
※フォルダ名に日本語があると、「\～」などで改行等とご認識され、うまくいかない時があるので、実行時にエラーが出たら疑ってください

【使い方】
PowerShell:
    python linkTomp4_improved_final.py

batファイルを実行する

1. YouTube URL を入力
2. 保存形式を選択（1: MP4動画, 2: MP3音声）

======================================================================
"""
import shutil
import subprocess
import sys
import re

from pathlib import Path#20251230　add 保存先を任意で選べるようにする
from tkinter import Tk, filedialog#20251230　add 保存先を任意で選べるようにする

# 保存先　# 0251230 del 保存先を任意で選べるようにする
#OUTPUT_TEMPLATE = (
#    r"C:\Users\admin\Desktop\cmd経由でYoutubeからmp4とmp3を選択してダウンロードする\%(title)s.%(ext)s"
#)

def normalize_youtube_url(url: str) -> str:
    """Shorts URL を watch?v= 形式へ変換"""
    m = re.match(r"https?://(www\.)?youtube\.com/shorts/([A-Za-z0-9_-]+)", url)
    if m:
        return f"https://www.youtube.com/watch?v={m.group(2)}"
    return url

def run(cmd: list[str]) -> None:
    """外部コマンド実行"""
    try:
        subprocess.run(cmd, check=True)
    except subprocess.CalledProcessError as e:
        sys.exit(f"コマンド失敗: {' '.join(cmd)}\n終了コード: {e.returncode}")
    except FileNotFoundError:
        sys.exit("エラー: yt-dlp または ffmpeg が見つかりません。PATH を確認してください。")
    except Exception as e:
        sys.exit(f"予期せぬエラー: {e}")

def main() -> None:
    # yt-dlp と ffmpeg の存在確認
    if shutil.which("yt-dlp") is None:
        sys.exit("エラー: yt-dlp が見つかりません。PATH を確認してください。")

    if shutil.which("ffmpeg") is None:
        sys.exit("エラー: ffmpeg が見つかりません。PATH を確認してください。")

    url = input("ダウンロードしたい YouTube の URL を入力してください: ").strip()
    if not url:
        sys.exit("URL が空です。終了します。")

    # 20251230(s) del 保存先を任意で選べるようにする
    # GUI 非表示
    Tk().withdraw()

    print("保存先フォルダを選択してください")
    output_dir = filedialog.askdirectory()
    if not output_dir:
        sys.exit("保存先が選択されなかったため終了します。")

    # 20251230(E) del 保存先を任意で選べるようにする

    output_dir = Path(output_dir)
    print("保存先フォルダ:", output_dir)# 20251230 add 保存先の表示

    output_template = str(output_dir / "%(title)s.%(ext)s")
    
    url = normalize_youtube_url(url)

    choice = input("保存形式を選んでください (1: MP4動画, 2: MP3音声): ").strip()

    common = ["yt-dlp", "-o", output_template]

    if choice == "1":
        cmd = common + [
            "-f", "bv*+ba/b",
            "--merge-output-format", "mp4",
            "--audio-format", "aac",
            "--recode-video", "mp4",
            url,
        ]
        run(cmd)
        print("音声互換性の高い MP4 をダウンロードしました。")

    elif choice == "2":
        cmd = common + [
            "-x",
            "--audio-format", "mp3",
            "--audio-quality", "0",
            url,
        ]
        run(cmd)
        print("MP3 音声をダウンロードしました。")

    else:
        sys.exit("不正な入力です。終了します。")

if __name__ == "__main__":
    main()
